﻿using System;
using System.Collections.Generic;
using System.Linq;
using Mdws2ORM.Maps.BaseFields;
using Mdws2ORM.Maps.Fields;
using Mdws2ORM.Core;
using Mdws2ORM.Maps.SingleMaps;

namespace Mdws2ORM.Maps
{
    public abstract class BaseEntityMap<T> : IEntityMap where T : class
    {
        private class Input
        {
            public readonly List<BaseFieldMap<T>> Fields = new List<BaseFieldMap<T>>();
            public readonly List<BaseMultipleFieldMap<T>> Multiples = new List<BaseMultipleFieldMap<T>>();
        }

        private bool isInit = false;
        private bool sortFields = true;

        public BaseFieldMap<T>[] GetFieldsMap;
        public BaseFieldMap<T>[] ListFieldsMap;
        public BaseFieldMap<T>[] GetForListFieldsMap;
        public BaseMultipleFieldMap<T>[] MultipleFieldsMap;

        private Input input = new Input();
        public QueryParam GetQueryParam { get; private set; }
        public QueryParam ListQueryParam { get; private set; }
        public QueryParam GetForListQueryParam { get; private set; }

        public bool SortFields
        {
            get
            {
                return sortFields;
            }
            set
            {
                sortFields = value;
            }
        }

        public IList<BaseFieldMap<T>> AllFields
        {
            get;
            private set;
        }

        private bool IsFieldUnique(string field)
        {
            if (input.Fields.Exists(e => e.Field == field))
                return false;

            if (input.Multiples.Exists(e => e.Field == field))
                return false;

            return true;
        }

        public void Map(BaseFieldMap<T> fieldMap)
        {
            if (!IsFieldUnique(fieldMap.Field))
                throw new ArgumentException();

            input.Fields.Add(fieldMap);
        }

        public void Map(BaseMultipleFieldMap<T> fieldMap)
        {
            if (!IsFieldUnique(fieldMap.Field))
                throw new ArgumentException();

            input.Multiples.Add(fieldMap);
        }

        public void Initialize(InitMapMediator initMediator)
        {
            if (isInit) return;
            isInit = true;
            MapFields();
            InitGetFields(initMediator);
            InitGetForListFields(initMediator);
            InitListFields(initMediator);
            InitMultipleFields(initMediator);
            this.AllFields = this.input.Fields.AsReadOnly();
            input = null;
        }

        private void InitGetFields(InitMapMediator initMediator)
        {
            GetFieldsMap = input
                .Fields
                .OrderBy(s => sortFields ? double.Parse(s.Field) : (double)input.Fields.IndexOf(s))
                .ToArray();

            GetFieldsMap.ForEach(s => s.Init(initMediator));

            string[] fields = GetFieldsMap.Select(s => s.Field).ToArray();
            string fieldParam = initMediator.MakeGetFieldsParam(fields);
            GetQueryParam = new QueryParam(File, fieldParam, fields);
        }

        private void InitListFields(InitMapMediator initMediator)
        {
            ListFieldsMap = input
               .Fields
               .Where(s => !s.FetchGetOnly)
               .OrderBy(s => sortFields ? double.Parse(s.Field) : (double)input.Fields.IndexOf(s))
               .ToArray();

            string[] fields = ListFieldsMap.Select(s => s.Field).ToArray();
            string fieldParam = initMediator.MakeListFieldsParam(fields);
            ListQueryParam = new QueryParam(File, fieldParam, fields);
        }

        private void InitGetForListFields(InitMapMediator initMediator)
        {
            GetForListFieldsMap = input.Fields
                                        .Where(s => s.FetchGetOnly)
                                        .OrderBy(s => sortFields ? double.Parse(s.Field) : (double)input.Fields.IndexOf(s))
                                        .ToArray();

            if (!GetForListFieldsMap.Any()) return;
            string[] fields = GetForListFieldsMap.Select(s => s.Field).ToArray();
            string fieldParam = initMediator.MakeGetFieldsParam(fields);
            GetForListQueryParam = new QueryParam(File, fieldParam, fields);
        }

        private void InitMultipleFields(InitMapMediator initMediator)
        {
            MultipleFieldsMap = input.Multiples.OrderBy(s => double.Parse(s.Field)).ToArray();
            MultipleFieldsMap.ForEach(s => s.Init(initMediator));
        }


        public abstract T NewEntity(string ien);
        public abstract string File { get; }
        protected abstract void MapFields();

        #region MapExtensions

        public void MapFreeText(string field, Action<T, string> mapAction)
        {
            Map(new FreeTextFieldMap<T>(field, mapAction));
        }

        public void MapDateTime(string field, Action<T, DateTime> mapAction)
        {
            Map(new DateTimeFieldMap<T>(field, mapAction));
        }

        public void MapBoolean(string field, Action<T, bool> mapAction)
        {
            Map(new BooleanFieldMap<T>(field, mapAction));
        }

        public void MapSetOfCodes(string field, Action<T, char> mapAction)
        {
            Map(new SetOfCodesFieldMap<T>(field, mapAction));
        }

        public void MapNumericInt(string field, Action<T, int> mapAction)
        {
            Map(new NumericIntFieldMap<T>(field, mapAction));
        }

        public void MapMultiple<V>(string field, Action<T, IEnumerable<V>> actionMap, SubEntityMap<V> subEntityMap = null)
            where V : class
        {
            Map(new MultipleFieldMap<T, V>(field, actionMap, subEntityMap));
        }

        public void MapMultipleSingle<V>(string field, Action<T, IEnumerable<V>> actionMap, SubEntityMap<Container<V>> subEntityMap)
        {
            if (subEntityMap == null)
                throw new ArgumentNullException("subEntityMap");

            Map(new MultipleFieldMap<T, Container<V>>(field, (t, v) => actionMap(t, v.Select(s => s.Item)), subEntityMap));
        }

        public void MapPointerToFile<V>(string field, Action<T, string> mapAction, Action<T, V> mapPointerAction, EntityMap<V> entityMap = null)
            where V : class
        {
            Map(new PointerToFileFieldMap<T, V>(field, mapAction, mapPointerAction, entityMap));
        }

        public void MapPointerToFileSingle<V>(string field, Action<T, string> mapAction, Action<T, V> mapPointerAction, EntityMap<Container<V>> entityMap)
        {
            if (entityMap == null)
                throw new ArgumentNullException("entityMap");

            Map(new PointerToFileFieldMap<T, Container<V>>(field, mapAction, (t, v) => mapPointerAction(t, v.Item), entityMap));
        }

        public void MapPointerToFile(string field, Action<T, string> mapAction)
        {
            Map(new PointerToFileFieldMap<T>(field, mapAction));
        }

        public void MapVariablePointer(string field, VariablePointerItemsBuilder<T> builder)
        {
            Map(new VariablePointerFieldMap<T>(field, builder.Items.ToArray()));
        }

        public void MapVariablePointer(string field, params VariablePointerFieldMap<T>.BaseItem[] items)
        {
            Map(new VariablePointerFieldMap<T>(field, items));
        }

        public void MapWordProcessing(string field, Action<T, string> mapAction)
        {
            Map(new WordProcessingFieldMap<T>(field, mapAction));
        }

        public void MapComputed(string field, Action<T, string> mapAction, bool getFetchOnly = false)
        {
            Map(new ComputedFieldMap<T>(field, mapAction, getFetchOnly));
        }

        public VariablePointerItemsBuilder<T> MakeVariablePointerItems()
        {
            return new VariablePointerItemsBuilder<T>();
        }

        #endregion
    }
}
